<?php
/**
 * Theme Framework functions and definitions
 *
 * Sets up the theme and provides some helper functions, which are used
 * in the theme as custom template tags. Others are attached to action and
 * filter hooks in WordPress to change core functionality.
 *
 * When using a child theme (see http://codex.wordpress.org/Theme_Development and
 * http://codex.wordpress.org/Child_Themes), you can override certain functions
 * (those wrapped in a function_exists() call) by defining them first in your child theme's
 * functions.php file. The child theme's functions.php file is included before the parent
 * theme's file, so the child theme functions would be used.
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are instead attached
 * to a filter or action hook.
 *
 * For more information on hooks, actions, and filters, @link http://codex.wordpress.org/Plugin_API
 *
 * @package CMSSuperHeroes
 * @subpackage CMS Theme
 * @since 1.0.0
 */

// Set up the content width value based on the theme's design and stylesheet.
if ( ! isset( $content_width ) )
	$content_width = 1170;
	
/**
 * CMS Theme setup.
 *
 * Sets up theme defaults and registers the various WordPress features that
 * CMS Theme supports.
 *
 * @uses load_theme_textdomain() For translation/localization support.
 * @uses add_editor_style() To add a Visual Editor stylesheet.
 * @uses add_theme_support() To add support for post thumbnails, automatic feed links,
 * 	custom background, and post formats.
 * @uses register_nav_menu() To add support for navigation menus.
 * @uses set_post_thumbnail_size() To set a custom post thumbnail size.
 *
 * @since 1.0.0
 */
function buildana_setup() {

	// load language.
	load_theme_textdomain( 'buildana' , get_template_directory() . '/languages' );

	// Adds title tag
	add_theme_support( "title-tag" );
	
	// Add woocommerce
	add_theme_support('woocommerce');
	 
	// Adds custom header
	add_theme_support( 'custom-header' );
	
	// Adds RSS feed links to <head> for posts and comments.
	add_theme_support( 'automatic-feed-links' );

	// This theme supports a variety of post formats.
	add_theme_support( 'post-formats', array( 'video', 'audio' , 'gallery', 'quote') );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menu( 'primary', esc_html__( 'Primary Menu', 'buildana' ) );

	/*
	 * This theme supports custom background color and image,
	 * and here we also set up the default background color.
	 */
	add_theme_support( 'custom-background', array('default-color' => 'e6e6e6',) );

	// This theme uses a custom image size for featured images, displayed on "standard" posts.
	  
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 370, 9999 ); // Unlimited height, soft crop
     
    $thubnail_size = [
    	'thumbnail_size_w'	=> 370,
		'thumbnail_size_h'	=> 370,
		'thumbnail_crop'	=> 1,
		'medium_size_w'		=> 740,
		'medium_size_h'		=> 440,
		'medium_crop'		=> 1,
		'large_size_w'		=> 1170,
		'large_size_h'		=> 590,
		'large_crop'		=> 1,
    ];
    foreach ($thubnail_size as $option => $value) {
    	if (get_option($option, '') != $value)
            update_option($option, $value);
    }
    add_image_size('buildana-370-256', 370, 256, true);	
    add_image_size('buildana-580-540', 580, 540, true);	
    add_image_size('buildana-540-430', 540, 430, true);
    add_image_size('buildana-540-380', 540, 380, true);	
    add_image_size('buildana-970-460', 970, 460, true);	
 
	add_editor_style( array( 'assets/css/editor-style.css' ) );
}

add_action( 'after_setup_theme', 'buildana_setup' );

/**
 * Call default WC single image gallery
 */
function buildana_wc_single_gallery()
{
    add_theme_support('wc-product-gallery-zoom');
    add_theme_support('wc-product-gallery-lightbox');
    add_theme_support('wc-product-gallery-slider'); 
}

add_action('after_setup_theme', 'buildana_wc_single_gallery');

/**
 * support shortcodes
 * @return array
 */
function buildana_shortcodes(){
	return array(
        'cms_fancybox',
		'cms_carousel',
		'cms_grid',
		'cms_fancybox_single',
        'cms_counter',
		'cms_counter_single',
		'cms_progressbar',
        'ef4_cms_grid',
	);
}

/**
 * Add new elements for VC
 * 
 * @author FOX
 */
add_action('vc_before_init', 'buildana_vc_before');

function buildana_vc_before(){
    
    require( get_template_directory() . '/vc_params/vc_custom.php' );
     
    if(!class_exists('CmsShortCode'))
        return ;
    
    require( get_template_directory() . '/inc/elements/cms_blog_list.php' );
    require( get_template_directory() . '/inc/elements/cms_project.php' );
    require( get_template_directory() . '/inc/elements/cms_custom_heading.php' );
    require( get_template_directory() . '/inc/elements/cms_history.php' );
    require( get_template_directory() . '/inc/elements/cms_testimonial_carousel.php' );
    require( get_template_directory() . '/inc/elements/cms_client.php' );
    require( get_template_directory() . '/inc/elements/cms_pricing.php' );
    require( get_template_directory() . '/inc/elements/cms_button.php' );
    require( get_template_directory() . '/inc/elements/cms_period.php' );
    require( get_template_directory() . '/inc/elements/cms_client_carousel.php' );
    require( get_template_directory() . '/inc/elements/cms_testimonial_vertical_carousel.php' );
    require( get_template_directory() . '/inc/elements/cms_what_doing.php' );
     
    add_filter('cms-shorcode-list', 'buildana_shortcodes');
     
}

/**
 * Custom params & remove VC Elements.
 * 
 * @author FOX
 */
add_action('vc_after_init', 'buildana_vc_after');

function buildana_vc_after() {
    require( get_template_directory() . '/vc_params/vc_column_text.php' ); 
    require( get_template_directory() . '/vc_params/vc_row.php' );
    require( get_template_directory() . '/vc_params/vc_column.php' );
    require( get_template_directory() . '/vc_params/vc_custom_heading.php' );
}

/* Add widgets */
require( get_template_directory() . '/inc/widgets/cms_business_hours.php' );
require( get_template_directory() . '/inc/widgets/recent_post_v2.php' );
require( get_template_directory() . '/inc/widgets/cms_gallery.php' );
require( get_template_directory() . '/inc/widgets/cms_certificates.php' );    
require( get_template_directory() . '/inc/widgets/cms_socials.php' );    
require( get_template_directory() . '/inc/widgets/cms_instagram.php' );
require( get_template_directory() . '/inc/widgets/cart_search/cart_search.php' );       

function buildana_generate_uiqueid( $length = 6 )
{
    return substr( md5( microtime() ), rand( 0, 26 ), $length );
}

/**
 * add google font
 */
function buildana_poppins() {
    $fonts_url = '';
    $poppins = _x( 'on', 'Poppins font: on or off', 'buildana' );
    if ( 'off' !== $poppins ) {
        $query_args = array(
        'family' =>  'Poppins:100,100i,200,200i,300,300i,400,400i,500,500i,600,600i,700,700i,800,800i,900,900i', 
        'subset' => urlencode( 'latin,latin-ext' )
        );
    }  
    $fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
    return esc_url_raw( $fonts_url );
}
function buildana_lato() {
    $fonts_url = '';
    $lato = _x('on','Lato font: on or off','buildana');
     if ( 'off' !== $lato ) {
        $query_args = array(
        'family' =>  'Lato:400,100,100italic,300,300italic,400italic,700,700italic,900,900italic', 
        'subset' => urlencode( 'latin,latin-ext' )
        );
      }
    $fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
    return esc_url_raw( $fonts_url );
}
function buildana_arimo() {
    $fonts_url = '';
    $arimo = _x('on','Arimo font: on or off','buildana');
     if ( 'off' !== $arimo ) {
        $query_args = array(
        'family' =>  'Arimo:400,400i,700,700i', 
        'subset' => urlencode( 'latin,latin-ext' )
        );
      }
    $fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
    return esc_url_raw( $fonts_url );
}
 
/**
 * Enqueue scripts and styles for front-end.
 * @author Fox
 * @since CMS SuperHeroes 1.0
 */
function buildana_front_end_scripts() {
    
	global $wp_styles, $opt_theme_options;
    
    $script_options = array(
	    'menu_sticky'=> '0',
        '1'         => 1,
	);
    if(isset($opt_theme_options['menu_sticky']) && $opt_theme_options['menu_sticky'] == '1'){
        $script_options['menu_sticky'] = $opt_theme_options['menu_sticky'];
    }
    /* Add Google font */
     
    wp_enqueue_style( 'buildana-poppins-font', buildana_poppins(), array(), null );
    wp_enqueue_style( 'buildana-lato-font', buildana_lato(), array(), null );
    wp_enqueue_style( 'buildana-arimo-font', buildana_arimo(), array(), null );
     
	/* Adds JavaScript Bootstrap. */
	wp_enqueue_script('bootstrap', get_template_directory_uri() . '/assets/js/bootstrap.min.js', array('jquery'), '3.3.2');
	
    wp_enqueue_script('bootstrap-select',get_template_directory_uri().'/assets/js/bootstrap-select.js',array('bootstrap'),'1.0');
    
    /* Adds JavaScript Bootstrap. */
	wp_enqueue_script('wow-effect', get_template_directory_uri() . '/assets/js/wow.min.js', array( 'jquery' ), '1.0.1', true);
    
    /* Adds magnific popup. */
    wp_enqueue_script('magnific-popup', get_template_directory_uri() . '/assets/js/jquery.magnific-popup.min.js', array( 'jquery' ), '1.0.0', true);
    
    /* Register isotope */
    wp_enqueue_script('isotope',get_template_directory_uri().'/assets/js/isotope.js',array('jquery'),'1.5.25',true);
    
    /* Add mousewheel js*/
    wp_enqueue_script('mousewheel', get_template_directory_uri() . '/assets/js/jquery.mousewheel.min.js', array( 'jquery' ), '1.0.0', true);   
    
	/* Add main.js */
	wp_register_script('buildana-main', get_template_directory_uri() . '/assets/js/main.js', array('jquery'), '1.0.0', true);
	 
	wp_localize_script('buildana-main', 'CMSOptions', $script_options);
	wp_enqueue_script('buildana-main');
     
	/* Add menu.js */
	wp_enqueue_script('buildana-menu', get_template_directory_uri() . '/assets/js/menu.js', array('jquery'), '1.0.0', true);
    
	/* Comment */
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) )
		wp_enqueue_script( 'comment-reply' );

	/** ----------------------------------------------------------------------------------- */
	
	/* Loads Bootstrap stylesheet. */
	wp_enqueue_style('bootstrap', get_template_directory_uri() . '/assets/css/bootstrap.min.css');
	
    wp_enqueue_style('bootstrap-select-css', get_template_directory_uri() . '/assets/css/bootstrap-select.css');
    
    /* Loads Animation. */
	wp_enqueue_style('animation', get_template_directory_uri() . '/assets/css/animate.css');
      
    /* Load magnific popup css*/
    wp_enqueue_style('magnific-popup-css', get_template_directory_uri() . '/assets/css/magnific-popup.css', array(), '1.0.1');
    
	/* Loads font awesome. */
	wp_enqueue_style('font-awesome', get_template_directory_uri() . '/assets/css/font-awesome.min.css');
    
    /* Loads Stroke gap font */
	wp_enqueue_style('font-stroke-7-icon', get_template_directory_uri() . '/assets/css/pe-icon-7-stroke.css');
     
    /* Loads rt-icon font */
	wp_enqueue_style('rt-icon', get_template_directory_uri() . '/assets/css/rt-icon.css');	  
    
    wp_enqueue_style('flaticon-icon', get_template_directory_uri() . '/assets/css/flaticon.css');
	/* Loads our main stylesheet. */
	wp_enqueue_style( 'buildana-style', get_stylesheet_uri());

	/* Loads the Internet Explorer specific stylesheet. */
	wp_enqueue_style( 'buildana-ie', get_template_directory_uri() . '/assets/css/ie.css');
	
	/* ie */
	$wp_styles->add_data( 'buildana-ie', 'conditional', 'lt IE 9' );
	
	/* Load static css*/
	wp_enqueue_style('buildana-static', get_template_directory_uri() . '/assets/css/static.css');
}

add_action( 'wp_enqueue_scripts', 'buildana_front_end_scripts' );

/**
 * load admin scripts.
 * 
 * @author FOX
 */
function buildana_admin_scripts(){

	/* Loads Bootstrap stylesheet. */
	wp_enqueue_style('font-awesome', get_template_directory_uri() . '/assets/css/font-awesome.min.css', array(), '4.3.0');
	wp_enqueue_style('font-glyphicons', get_template_directory_uri() . '/assets/css/glyphicons.css', array(), '3.0.0'); 
    wp_enqueue_style('font-rt-icon2', get_template_directory_uri() . '/assets/css/rt-icon.css', array(), '1.0.0'); 
    wp_enqueue_style('font-flaticon', get_template_directory_uri() . '/assets/css/flaticon.css', array(), '1.0.0'); 
    wp_enqueue_style('buildana-admin-css', get_template_directory_uri() . '/assets/css/admin-style.css', array(), '1.0.0');
    
	$screen = get_current_screen();

	/* load js for edit post. */
	if($screen->post_type == 'post'){
		/* post format select. */
		wp_enqueue_script('post-format', get_template_directory_uri() . '/assets/js/post-format.js', array(), '1.0.0', true);
	}
    
    wp_enqueue_script( 'buildana-admin', get_template_directory_uri() . '/assets/js/admin.js', array( 'jquery', 'jquery-ui-core', 'jquery-ui-sortable', 'media-upload' ) );
    wp_localize_script( 'buildana-admin', 'RedExpMediaLocalize', array(
        'add_video' => esc_html__( 'Add Video', 'buildana' ),
        'add_audio' => esc_html__( 'Add Audio', 'buildana' ),
        'add_images' => esc_html__( 'Add Image(s)', 'buildana' ),
        'add_image' => esc_html__( 'Add Image', 'buildana' )
    ) );
    
}

add_action( 'admin_enqueue_scripts', 'buildana_admin_scripts' );

/**
 * Register sidebars.
 *
 * Registers our main widget area and the front page widget areas.
 *
 * @since Fox
 */
function buildana_widgets_init() {

	global $opt_theme_options;

	register_sidebar( array(
		'name' => esc_html__( 'Main Sidebar', 'buildana' ),
		'id' => 'sidebar-1',
		'description' => esc_html__( 'Appears on posts and pages except the optional Front Page template, which has its own widgets', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
       
    register_sidebar( array(
		'name' => esc_html__( 'Header top 1 left Side', 'buildana' ),
		'id' => 'header-top-1-left',
		'description' => esc_html__( 'Appears on the top left of header', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
    register_sidebar( array(
		'name' => esc_html__( 'Header top 1 right Side', 'buildana' ),
		'id' => 'header-top-1-right',
		'description' => esc_html__( 'Appears on the top right of header', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
	register_sidebar( array(
		'name' => esc_html__( 'Header top 2 left Side', 'buildana' ),
		'id' => 'header-top-2-left',
		'description' => esc_html__( 'Appears on the top left of header', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
    register_sidebar( array(
		'name' => esc_html__( 'Header top 2 right Side', 'buildana' ),
		'id' => 'header-top-2-right',
		'description' => esc_html__( 'Appears on the top right of header', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
    
    register_sidebar( array(
		'name' => esc_html__( 'Header middle 1 left Side', 'buildana' ),
		'id' => 'header-middle-1-left',
		'description' => esc_html__( 'Appears on the middle left of header', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
    register_sidebar( array(
		'name' => esc_html__( 'Header middle 1 right Side', 'buildana' ),
		'id' => 'header-middle-1-right',
		'description' => esc_html__( 'Appears on the middle right of header', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
    register_sidebar( array(
		'name' => esc_html__( 'Header cart search', 'buildana' ),
		'id' => 'header-cart-search',
		'description' => esc_html__( 'Appears on the right of menu', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
    register_sidebar( array(
		'name' => esc_html__( 'Header cart search 2', 'buildana' ),
		'id' => 'header-cart-search2',
		'description' => esc_html__( 'Appears on the right of menu', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) ); 
    if (class_exists('WooCommerce')) {
        register_sidebar(array(
            'name'          => esc_html__('WooCommerce Sidebar', 'buildana'),
            'id'            => 'sidebar-shop',
            'description'   => esc_html__('Appears in WooCommerce Archive page', 'buildana'),
            'before_widget' => '<aside id="%1$s" class="widget %2$s">',
            'after_widget'  => '</aside>',
            'before_title'  => '<h3 class="wg-title"><span>',
            'after_title'   => '</span></h4>',
        ));
    } 
    register_sidebar( array(
		'name' => esc_html__( 'Newsleter', 'buildana' ),
		'id' => 'newsleter',
		'description' => esc_html__( 'Appears on the top of header', 'buildana' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget' => '</aside>',
		'before_title' => '<h3 class="wg-title">',
		'after_title' => '</h3>',
	) );
    if(!empty($opt_theme_options['footer-top-column'])) {
		for($i = 1 ; $i <= $opt_theme_options['footer-top-column'] ; $i++){
			register_sidebar(array(
				'name' => sprintf(esc_html__('Footer Top - col %s', 'buildana'), $i),
				'id' => 'sidebar-footer-top-col' . $i,
				'description' => esc_html__('Appears on posts and pages except the optional Front Page template, which has its own widgets', 'buildana'),
				'before_widget' => '<aside id="%1$s" class="widget %2$s">',
				'after_widget' => '</aside>',
				'before_title' => '<h3 class="wg-title">',
				'after_title' => '</h3>',
			));
		}
	}  
       
}

add_action( 'widgets_init', 'buildana_widgets_init' );

/**
 * Display navigation to next/previous comments when applicable.
 *
 * @since 1.0.0
 */
function buildana_comment_nav() {
    // Are there comments to navigate through?
    if ( get_comment_pages_count() > 1 && get_option( 'page_comments' ) ) :
    ?>
	<nav class="navigation comment-navigation" role="navigation">
		<h2 class="screen-reader-text"><?php esc_html_e( 'Comment navigation', 'buildana' ); ?></h2>
		<div class="nav-links">
			<?php
				if ( $prev_link = get_previous_comments_link( esc_html__( 'Older Comments', 'buildana' ) ) ) :
					printf( '<div class="nav-previous">%s</div>', $prev_link );
				endif;

				if ( $next_link = get_next_comments_link( esc_html__( 'Newer Comments', 'buildana' ) ) ) :
					printf( '<div class="nav-next">%s</div>', $next_link );
				endif;
			?>
		</div><!-- .nav-links -->
	</nav><!-- .comment-navigation -->
	<?php
	endif;
}

/**
 * Display navigation to next/previous set of posts when applicable.
 *
 * @since 1.0.0
 */
function buildana_paging_nav() {
    // Don't print empty markup if there's only one page.
	if ( $GLOBALS['wp_query']->max_num_pages < 2 ) {
		return;
	}

	$paged        = get_query_var( 'paged' ) ? intval( get_query_var( 'paged' ) ) : 1;
	$pagenum_link = html_entity_decode( get_pagenum_link() );
	$query_args   = array();
	$url_parts    = explode( '?', $pagenum_link );

	if ( isset( $url_parts[1] ) ) {
		wp_parse_str( $url_parts[1], $query_args );
	}

	$pagenum_link = remove_query_arg( array_keys( $query_args ), $pagenum_link );
	$pagenum_link = trailingslashit( $pagenum_link ) . '%_%';

	// Set up paginated links.
	$links = paginate_links( array(
			'base'     => $pagenum_link,
			'total'    => $GLOBALS['wp_query']->max_num_pages,
			'current'  => $paged,
			'mid_size' => 1,
			'add_args' => array_map( 'urlencode', $query_args ),
			'prev_text' => esc_html('Prev post','buildana'),
			'next_text' => esc_html('Next post','buildana'),
	) );

	if ( $links ) :

	?>
	<nav class="navigation paging-navigation clearfix text-center">
			<div class="pagination loop-pagination">
                <div class="pagin-inner clearfix">
				    <?php echo wp_kses_post($links); ?>
                </div>
			</div><!-- .pagination -->
	</nav><!-- .navigation -->
	<?php
	endif;
}

/**
* Display navigation to next/previous post when applicable.
*
* @since 1.0.0
*/
function buildana_post_nav() {
    global $post,$opt_theme_options;
    if(!isset($opt_theme_options['single_post_nav']) ||  ( isset($opt_theme_options['single_post_nav']) && !$opt_theme_options['single_post_nav']))
        return;

    // Don't print empty markup if there's nowhere to navigate.
    $previous = ( is_attachment() ) ? get_post( $post->post_parent ) : get_adjacent_post( false, '', true );
    $next     = get_adjacent_post( false, '', false );

    if ( ! $next && ! $previous )
        return;
    ?>
	<nav class="navigation post-navigation" role="navigation">
		<div class="row nav-links clearfix">
			<?php
			$prev_post = get_previous_post();
			if (!empty( $prev_post )): 
            $thumbnail_bg = get_the_post_thumbnail_url($prev_post->ID, 'thumbnail');
            $style='';
            if ( $thumbnail_bg ) {
                $style = 'style="background-image:url('.$thumbnail_bg.'); background-size: cover;"';
            }
            ?>
            <div class="col-sm-6">
                <div class="post-nav-wrap text-center" <?php echo ''.$style;?>>
                <a class="post-prev left" href="<?php echo get_permalink( $prev_post->ID ); ?>"><?php echo esc_html__('Previous','buildana'); ?></a>
                <h3><a href="<?php echo get_permalink( $prev_post->ID ); ?>"><?php echo get_the_title($prev_post->ID);?></a></h3>
                </div>  
            </div>
			<?php endif; ?>
            
			<?php
			$next_post = get_next_post();
			if ( is_a( $next_post , 'WP_Post' ) ) { 
    			$thumbnail_bg = get_the_post_thumbnail_url($next_post->ID, 'thumbnail');
                $style='';
                if ( $thumbnail_bg ) {
                    $style = 'style="background-image:url('.$thumbnail_bg.'); background-size: cover;"';
                }
                 ?>
                <div class="col-sm-6 text-right">
                    <div class="post-nav-wrap text-center" <?php echo ''.$style;?>>
    			     <a class="post-next right" href="<?php echo get_permalink( $next_post->ID ); ?>"><?php echo esc_html__('Next','buildana'); ?></a>
                     <h3><a href="<?php echo get_permalink( $next_post->ID ); ?>"><?php echo get_the_title($next_post->ID);?></a></h3>
                    </div>  
                </div>  
			<?php } ?>

		</div><!-- .nav-links -->
	</nav><!-- .navigation -->
	<?php
}

function buildana_project_nav() {
    global $post;
    
    // Don't print empty markup if there's nowhere to navigate.
    $previous = ( is_attachment() ) ? get_post( $post->post_parent ) : get_adjacent_post( false, '', true );
    $next     = get_adjacent_post( false, '', false );

    if ( ! $next && ! $previous )
        return;
    ?>
	<nav class="navigation pro-navigation">
		<div class="row nav-links clearfix">
			<?php
			$prev_post = get_previous_post();
			if (!empty( $prev_post )): 
             
            ?>
            <div class="col-sm-6">
                <div class="pro-nav-wrap text-right">
                    <a class="pro-prev left btn btn-default" href="<?php echo get_permalink( $prev_post->ID ); ?>"><?php echo esc_html__('Previous','buildana'); ?></a>
                </div>  
            </div>
			<?php endif; ?>
			<?php
			$next_post = get_next_post();
			if ( is_a( $next_post , 'WP_Post' ) ) { 
                 ?>
                <div class="col-sm-6 text-left">
                    <div class="pro-nav-wrap">
    			         <a class="pro-next right btn btn-default" href="<?php echo get_permalink( $next_post->ID ); ?>"><?php echo esc_html__('Next','buildana'); ?></a>
                    </div>  
                </div>  
			<?php } ?>

		</div> 
	</nav> 
	<?php
}

function buildana_product_nav() {
    global $post; 
    
    $previous = ( is_attachment() ) ? get_post( $post->post_parent ) : get_adjacent_post( false, '', true );
    $next     = get_adjacent_post( false, '', false );

    if ( ! $next && ! $previous )
        return;
    ?>
	<nav class="navigation product-navigation">
		<div class="nav-links clearfix">
			<?php
			$prev_post = get_previous_post();
			if (!empty( $prev_post )): 
            ?>
            <a class="pro-prev left btn btn-default" href="<?php echo get_permalink( $prev_post->ID ); ?>"><?php echo esc_html__('Previous','buildana'); ?></a>   
			<?php endif; ?>
			<?php
			$next_post = get_next_post();
			if ( is_a( $next_post , 'WP_Post' ) ) { 
                ?>
    			<a class="pro-next right btn btn-default" href="<?php echo get_permalink( $next_post->ID ); ?>"><?php echo esc_html__('Next','buildana'); ?></a>   
			<?php } ?>
		</div> 
	</nav> 
	<?php
}
 

/* Add Custom Comment */
function buildana_comment($comment, $args, $depth) {
	$GLOBALS['comment'] = $comment;
	extract($args, EXTR_SKIP);

	if ( 'div' == $args['style'] ) {
		$tag = 'div';
		$add_below = 'comment';
	} else {
		$tag = 'li';
		$add_below = 'div-comment';
	}
    ?>
    <<?php echo esc_attr($tag) ?> <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ) ?> id="comment-<?php comment_ID() ?>">
    <?php if ( 'div' != $args['style'] ) : ?>
    <div id="div-comment-<?php comment_ID() ?>" class="comment-body clearfix">
    <?php endif; ?>
        <div class="comment-author-image vcard">
        	<?php echo get_avatar( $comment, 90 ); ?>
        </div>
    
        <div class="comment-main">
        
            <div class="comment-meta commentmetadata">
                <?php
        		echo '<span class="comment-author">'. get_comment_author_link() .'</span>';
                 
                echo '<span class="comment-date">';
                    $comment_date = get_comment_date('',get_comment_ID()); 
            	    echo esc_html__('Posted','buildana').' '.esc_attr($comment_date);  
            	echo '</span>';
                ?>
                <span class="comment-reply">
        		  <?php comment_reply_link( array_merge( $args, array( 'add_below' => $add_below, 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
        		</span>	
        	</div>
             
        </div>
        
        <div class="comment-content">
            <?php comment_text(); ?>
    	</div>
        <?php if ( $comment->comment_approved == '0' ) : ?>
        	<em class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.' , 'buildana'); ?></em>
        <?php endif; ?>

    <?php if ( 'div' != $args['style'] ) : ?>
    </div>
    <?php endif; ?>
    <?php
}
 
 

/**
 * Move comment form field to bottom
 */
function buildana_comment_field_to_bottom( $fields ) {
    $comment_field = $fields['comment'];
    unset( $fields['comment'] );
    $fields['comment'] = $comment_field;
    return $fields;
}
add_filter( 'comment_form_fields', 'buildana_comment_field_to_bottom' ); 

/**
 * limit words
 */

function buildana_limit_words($string) {
    global $opt_theme_options;
    if(isset($opt_theme_options['excerpt_length']) && !empty($opt_theme_options['excerpt_length']) && (int) $opt_theme_options['excerpt_length'] > 0){
        $word_limit =  $opt_theme_options['excerpt_length'];
        if(is_sticky()) $word_limit = 22;
        $words = explode(' ', $string, ($word_limit + 1));
        if (count($words) > $word_limit) {
            array_pop($words);
        }
        return implode(' ', $words).'...';
    }else{
        return $string.'...';
    }
}

function buildana_grid_limit_words($string, $word_limit) {
    $words = explode(' ', $string, ($word_limit + 1));
    if (count($words) > $word_limit) {
        array_pop($words);
    }
    return implode(' ', $words).'...';
}

function buildana_project_limit_words($string, $word_limit) {
    $words = explode(' ', $string, ($word_limit + 1));
    if (count($words) > $word_limit) {
        array_pop($words);
    }
    return implode(' ', $words);
}

/* Remove [...] after excerpt text */
function buildana_new_excerpt_more( $more ) {
    return '';
}
add_filter('excerpt_more', 'buildana_new_excerpt_more');

/* Custom excerpt length */
function buildana_custom_excerpt_length( $length ) { 
	return 75;
}

add_filter( 'excerpt_length', 'buildana_custom_excerpt_length', 999 );

/* core functions. */
require_once( get_template_directory() . '/inc/functions.php' );

/**
 * theme actions.
 */

/* add footer back to top. */
add_action('wp_footer', 'buildana_footer_back_to_top');
 
function buildana_check_plugin_version($class, $plugin_file_path){
	if( !function_exists('get_plugin_data') ){
	    require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
	}
    $plugin = class_exists($class) ? get_plugin_data(WP_PLUGIN_DIR.'/'.$plugin_file_path) : ['Version'=> '0'];
    $plugin['Version'] = str_replace('.', '', $plugin['Version']);
    return (int)$plugin['Version'];
}
function buildana_admin_notice(){
    $ef4_version = buildana_check_plugin_version('EF4Framework','ef4-Framework/ef4-framework.php');
    if(class_exists('EF4Framework') && $ef4_version < 210){
        ?>
        <div class="notice notice-error ef4-notice">
            <?php  ?>
            <p>
                <strong>
                    <?php echo esc_html__('EF4 Framework plugin using is out of date.','buildana');?>
                </strong>
            </p>
            <p>
                <?php printf(__('<strong><a href="%s">Click here</a></strong> to Deactive and then Delete EF4 Framework to update it to latest version','buildana'),
                    esc_url(admin_url('plugins.php'))
                ); 
                ?>
            </p>
        </div>
        <?php
    }
}
add_action('admin_notices', 'buildana_admin_notice', 0);

/**
 * Check framework/sytem plugin version to compatible with theme version
 * Action Deactive plugin
 * @since 2.0
*/
function deactivate_plugin_conditional() {
    $ef4_version = buildana_check_plugin_version('EF4Framework', 'ef4-Framework/ef4-framework.php');
    if ( class_exists('EF4Framework') && $ef4_version < 210 ) {
        deactivate_plugins('ef4-Framework/ef4-framework.php');    
    }
}
add_action( 'admin_init', 'deactivate_plugin_conditional' );

/**
 * Add Admin style
*/
add_action('admin_enqueue',function(){
	$admin_style = '<style>.ef4-notice strong{font-size: 30px;color: red;}</style>';
	return $admin_style;
});
/**
 * Support GutenBerg
 * @since 2.0
*/
require( get_template_directory() . '/inc/gutenberg/gutenberg.php' );
 
function buildana_html($html){
    return $html;
}

/* Remvoe EF4 Settings */
add_filter('ef4_show_settings_menu', '__return_false');